/*
  chat subsystem
  written by alexander yaworsky
  september '99
*/

#include <windows.h>

#include "crc.h"
#include "des.h"
#include "paths.h"
#include "stdlib.h"
#include "stringlist.h"
#include "chat.h"
#include "executive.h"
#include "errors.h"
#include "network.h"
#include "switches.h"


#define VROOMSZ    8192
#define NVROOMSZ   2048

static CHATDATA VRoom, NVRoom;
static volatile LONG InitDone = FALSE;
static char DesKey[] = "G`5I=;*|";

/*
   cyclic buffer operations
*/

static unsigned char GetNextByte( CHATDATA* Room )
  {
    unsigned char  v;

    if( Room->Head == Room->Tail ) return 0;
    v = Room->Buf[ Room->Head++ ];
    if( Room->Head == Room->BufSz ) Room->Head = 0;
    return v;
  }

static void GetMsgHeader( CHATDATA* Room, ULONG* Index, WORD* Date, WORD* Time )
  {
    unsigned char  a, b, c, d;

    a = GetNextByte( Room );
    b = GetNextByte( Room );
    c = GetNextByte( Room );
    d = GetNextByte( Room );
    *Index = a + (b << 8) + (c << 16) + (d << 24);
    a = GetNextByte( Room );
    b = GetNextByte( Room );
    *Date = a + (b << 8);
    a = GetNextByte( Room );
    b = GetNextByte( Room );
    *Time = a + (b << 8);
  }

static int PeekMsgSize( CHATDATA* Room )
  {
    int  i;
    CHATDATA  Rm;

    CopyMemory( &Rm, Room, sizeof( CHATDATA ) ); // don't modify room
    for( i = 0; GetNextByte( &Rm ) != 0; i++ );
    return i;
  }

static void PeekMsgData( CHATDATA* Room, char* Dest )
  {
    char  ch;
    CHATDATA  Rm;

    CopyMemory( &Rm, Room, sizeof( CHATDATA ) ); // don't modify room
    do {
      ch = GetNextByte( &Rm );
      *Dest = ch;
      Dest++;
    } while( ch != 0 );
  }

static void SkipMsgData( CHATDATA* Room )
  {
    while( GetNextByte( Room ) != 0 );
  }

static void SetNextByte( CHATDATA* Room, unsigned char Value )
  {
    int i;

    Room->Buf[ Room->Tail++ ] = Value;
    if( Room->Tail == Room->BufSz ) Room->Tail = 0;
    if( Room->Head == Room->Tail ) {
      // rewind 1 character
      if( Room->Tail == 0 ) Room->Tail = Room->BufSz;
      Room->Tail--;
      // erase latest message
      for( i = 0; i < 8; i++ ) GetNextByte( Room );
      SkipMsgData( Room );
      // put character again
      Room->Buf[ Room->Tail++ ] = Value;
      if( Room->Tail == Room->BufSz ) Room->Tail = 0;
    }
  }

/*
   chat room operations
*/

void LockRoom( CHATDATA* Room )
  {
    //EnterCriticalSection( &Room->CSect );
    WaitForSingleObject( Room->CSect, INFINITE );
  }

void UnlockRoom( CHATDATA* Room )
  {
    //LeaveCriticalSection( &Room->CSect );
    ReleaseMutex( Room->CSect );
  }

void AddChatMessage( CHATDATA* Room, char* Msg )
  {
    int    i, Tail;
    char   ch;
    WORD   d, t;
    FILETIME    Ftm;
    SYSTEMTIME  Stm;

    GetLocalTime( &Stm );
    SystemTimeToFileTime( &Stm, &Ftm );
    FileTimeToDosDateTime( &Ftm, &d, &t );
    Tail = Room->Tail; // save tail
    SetNextByte( Room, (unsigned char) Room->Index );
    SetNextByte( Room, (unsigned char) (Room->Index >> 8) );
    SetNextByte( Room, (unsigned char) (Room->Index >> 16) );
    SetNextByte( Room, (unsigned char) (Room->Index >> 24) );
    SetNextByte( Room, (unsigned char) d );
    SetNextByte( Room, (unsigned char) (d >> 8) );
    SetNextByte( Room, (unsigned char) t );
    SetNextByte( Room, (unsigned char) (t >> 8) );
    Room->Index++;
    for(;;) {
      ch = *Msg; Msg++;
      Room->Buf[ Room->Tail++ ] = ch;
      if( Room->Tail == Room->BufSz ) Room->Tail = 0;
      if( Room->Head == Room->Tail ) {
        // erase latest message or truncate
        if( Room->Head == Tail ) {
          // truncate - rewind 2 characters and put 0
          for( i = 0; i < 2; i++ ) {
            if( Room->Tail == 0 ) Room->Tail = Room->BufSz;
            Room->Tail--;
          }
          SetNextByte( Room, 0 );
          break;
        }
        else {
          // rewind 1 character
          if( Room->Tail == 0 ) Room->Tail = Room->BufSz;
          Room->Tail--;
          // erase latest msg
          for( i = 0; i < 8; i++ ) GetNextByte( Room );
          SkipMsgData( Room );
          // put character again
          Room->Buf[ Room->Tail++ ] = ch;
          if( Room->Tail == Room->BufSz ) Room->Tail = 0;
        }
      }
      if( ch == 0 ) break;
    }
  }

void GetChatMessage( CHATDATA* Room,
                     ULONG Index, // if index 0, all msgs
                     STRINGLIST* Result )
  {
    int    msz;
    ULONG  i;
    WORD   d, t;
    char   *Msg;
    FILETIME    Ftm;
    SYSTEMTIME  Stm;
    CHATDATA    Rm;

    CopyMemory( &Rm, Room, sizeof( CHATDATA ) ); // don't modify room
    while( Rm.Head != Rm.Tail ) {
      GetMsgHeader( &Rm, &i, &d, &t );
      msz = PeekMsgSize( &Rm );
      if( i >= Index ) {
        Msg = (char*) LocalAlloc( LPTR, msz + 64 );
        if( Msg != NULL ) {
          DosDateTimeToFileTime( d, t, &Ftm );
          FileTimeToSystemTime( &Ftm, &Stm );
          wsprintf( Msg, "%11d %02d:%02d:%02d %02d/%02d/%4d ",
                    i, (int) Stm.wHour, (int) Stm.wMinute, (int) Stm.wSecond,
                    (int) Stm.wDay, (int) Stm.wMonth, (int) Stm.wYear );
          PeekMsgData( &Rm, Msg + lstrlen( Msg ) );
          AddToStringList( Result, Msg );
          LocalFree( Msg );
        }
      }
      SkipMsgData( &Rm );
    }
  }

void ClearRoom( CHATDATA* Room )
  {
    Room->Head = 0;
    Room->Tail = 0;
  }

BOOL InitRoom( CHATDATA* Room, int BufSz )
  {
    Room->Buf = (char*) LocalAlloc( LPTR, BufSz );
    if( Room->Buf == NULL ) return FALSE;
    Room->Head = 0;
    Room->Tail = 0;
    Room->BufSz = BufSz;
    Room->Index = 1;
    //InitializeCriticalSection( &Room->CSect );
    if( (Room->CSect = CreateMutex( NULL, FALSE, NULL )) == NULL ) {
      LocalFree( Room->Buf );
      return FALSE;
    }
    return TRUE;
  }

void DeinitRoom( CHATDATA* Room )
  {
    LocalFree( Room->Buf );
    //DeleteCriticalSection( &Room->CSect );
    CloseHandle( Room->CSect );
  }

static BOOL SetRoomSize( CHATDATA* Room, int BufSz )
  {
    char  *NewBuf;

    NewBuf = (char*) LocalReAlloc( (HLOCAL) Room->Buf, BufSz, 0 );
    if( NewBuf == NULL ) return FALSE;
    Room->Buf = NewBuf;
    Room->Head = 0;
    Room->Tail = 0;
    Room->BufSz = BufSz;
    return TRUE;
  }

static void LoadRoom( CHATDATA* Room )
  {
    HKEY   KeyH;
    LPBYTE VData;
    DWORD  VType, VSize;
    ULONG  Crc32;
    int    i, BufSz;
    DESDATA  DesCtx;

    if( RegOpenKeyEx( HKEY_LOCAL_MACHINE, RegKeyHome,
                      0, KEY_ALL_ACCESS, &KeyH ) != ERROR_SUCCESS ) return;
    if( RegQueryValueEx( KeyH, RegValChatData, 0,
                         &VType, NULL, &VSize ) == ERROR_SUCCESS ) {
      VData = LocalAlloc( LPTR, VSize );
      if( VData != NULL ) {
        if( RegQueryValueEx( KeyH, RegValChatData, 0,
                             &VType, VData, &VSize ) == ERROR_SUCCESS ) {
          CopyMemory( DesCtx.Key, DesKey, 8 );
          DesCtx.Iterations = 16;
          DesCtx.ModeCBC = 1;
          DesInit( &DesCtx );
          DesDecrypt( &DesCtx );
          DesCrypt( &DesCtx, VData, VSize );
          CRC32_Init( Crc32 );
          for( i = 0; i < VSize - 4; i++ ) CRC32_Upd( Crc32, VData[ i ] );
          if( Crc32 == (((unsigned)VData[i]) & 255) +
                       ((((unsigned)VData[i+1]) & 255) << 8) +
                       ((((unsigned)VData[i+2]) & 255) << 16) +
                       ((((unsigned)VData[i+3]) & 255) << 24) ) {
            BufSz = (((unsigned)VData[0]) & 255) +
                    ((((unsigned)VData[1]) & 255) << 8) +
                    ((((unsigned)VData[2]) & 255) << 16) +
                    ((((unsigned)VData[3]) & 255) << 24);
            if( BufSz <= Room->BufSz && VSize - 12 < BufSz ) {
              Room->Head = 0;
              Room->Tail = VSize - 12;
              Room->Index = (((unsigned)VData[4]) & 255) +
                            ((((unsigned)VData[5]) & 255) << 8) +
                            ((((unsigned)VData[6]) & 255) << 16) +
                            ((((unsigned)VData[7]) & 255) << 24);
              CopyMemory( Room->Buf, VData + 8, VSize - 12 );
            }
          }
        }
      }
      LocalFree( VData );
    }
    RegCloseKey( KeyH );
  }

static void SaveRoom( CHATDATA* Room )
  {
    HKEY   KeyH;
    LPBYTE VData;
    DWORD  VSize;
    ULONG  Crc32;
    int    i;
    DESDATA  DesCtx;

    if( RegOpenKeyEx( HKEY_LOCAL_MACHINE, RegKeyHome,
                      0, KEY_ALL_ACCESS, &KeyH ) != ERROR_SUCCESS ) return;
    VSize = Room->Tail - Room->Head;
    if( Room->Head > Room->Tail ) VSize += Room->BufSz;
    VSize += 12;
    VData = LocalAlloc( LPTR, VSize );
    if( VData != NULL ) {
      VData[0] = (char) Room->BufSz;
      VData[1] = (char) (Room->BufSz >> 8);
      VData[2] = (char) (Room->BufSz >> 16);
      VData[3] = (char) (Room->BufSz >> 24);
      VData[4] = (char) Room->Index;
      VData[5] = (char) (Room->Index >> 8);
      VData[6] = (char) (Room->Index >> 16);
      VData[7] = (char) (Room->Index >> 24);
      if( Room->Head != Room->Tail ) {
        if( Room->Head < Room->Tail )
          CopyMemory( VData + 8, Room->Buf + Room->Head, Room->Tail - Room->Head );
        else {
          i = Room->BufSz - Room->Head;
          CopyMemory( VData + 8, Room->Buf + Room->Head, i );
          CopyMemory( VData + 8 + i, Room->Buf, Room->Tail );
        }
      }
      CRC32_Init( Crc32 );
      for( i = 0; i < VSize - 4; i++ ) CRC32_Upd( Crc32, VData[ i ] );
      VData[i] = (char) Crc32;
      VData[i+1] = (char) (Crc32 >> 8);
      VData[i+2] = (char) (Crc32 >> 16);
      VData[i+3] = (char) (Crc32 >> 24);
      CopyMemory( DesCtx.Key, DesKey, 8 );
      DesCtx.Iterations = 16;
      DesCtx.ModeCBC = 1;
      DesInit( &DesCtx );
      DesEncrypt( &DesCtx );
      DesCrypt( &DesCtx, VData, VSize );
      RegSetValueEx( KeyH, RegValChatData, 0, REG_BINARY, VData, VSize );
      LocalFree( VData );
    }
    RegCloseKey( KeyH );
  }

BOOL InitChat()
  {
    static volatile LONG Flag = 0;

    if( InitDone ) return TRUE;
    if( InterlockedExchange( (LPLONG) &Flag, 1 ) != 0 ) {
      while( InterlockedExchange( (LPLONG) &Flag, 1 ) != 0 ) Sleep(0);
      InterlockedExchange( (LPLONG) &Flag, 0 );
      return (BOOL) InitDone;
    }
    if( ! InitRoom( &VRoom, VROOMSZ ) ) return FALSE;
    if( InitRoom( &NVRoom, NVROOMSZ ) ) {
      LoadRoom( &NVRoom );
      InitDone = TRUE;
      InterlockedExchange( (LPLONG) &Flag, 0 );
      return TRUE;
    }
    DeinitRoom( &VRoom );
    InterlockedExchange( (LPLONG) &Flag, 0 );
    return FALSE;
  }


FUNCDEF( ChatClr )
  {
    long  NV = 0;

    if( ! InitChat() ) {
      SetReply( Result, ERR_CHAT_INIT, "" );
      return 1;
    }
    if( Params->Count > 1 ) NV = Strtoul( Params->Value[1], NULL, 0 );
    if( NV ) {
      LockRoom( &NVRoom );
      ClearRoom( &NVRoom );
      SaveRoom( &NVRoom );
      UnlockRoom( &NVRoom );
    }
    else {
      LockRoom( &VRoom );
      ClearRoom( &VRoom );
      UnlockRoom( &VRoom );
    }
    SetReply( Result, ERR_NO_ERROR, "" );
    return 1;
  }

FUNCDEF( ChatRD )
  {
    ULONG  i = 0;

    if( ! InitDone ) {
      SetReply( Result, ERR_NO_ERROR, "" );
      return 1;
    }
    if( ! InitChat() ) {
      SetReply( Result, ERR_CHAT_INIT, "" );
      return 1;
    }
    if( Params->Count > 1 ) i = Strtoul( Params->Value[1], NULL, 0 );
    SetReply( Result, ERR_NO_ERROR, "" );
    LockRoom( &VRoom );
    GetChatMessage( &VRoom, i, Result );
    UnlockRoom( &VRoom );
    return 1;
  }

FUNCDEF( ChatRDnv )
  {
    ULONG  i = 0;

    if( ! InitChat() ) {
      SetReply( Result, ERR_CHAT_INIT, "" );
      return 1;
    }
    if( Params->Count > 1 ) i = Strtoul( Params->Value[1], NULL, 0 );
    SetReply( Result, ERR_NO_ERROR, "" );
    LockRoom( &NVRoom );
    GetChatMessage( &NVRoom, i, Result );
    UnlockRoom( &NVRoom );
    return 1;
  }

FUNCDEF( ChatSize )
  {
    ULONG  Kbytes = 0;
    BOOL   Rc;
    char   Buf[ 128 ];

    if( ! InitChat() ) {
      SetReply( Result, ERR_CHAT_INIT, "" );
      return 1;
    }
    if( Params->Count > 1 ) {
      Kbytes = Strtoul( Params->Value[1], NULL, 0 );
      if( Kbytes == 0 || Kbytes > 64 ) {
        SetReply( Result, ERR_CHAT_SIZE, "" );
        return 1;
      }
    }
    if( Kbytes != 0 ) {
      LockRoom( &VRoom );
      Rc = SetRoomSize( &VRoom, Kbytes * 1024 );
      UnlockRoom( &VRoom );
      if( ! Rc ) {
        SetReply( Result, ERR_OUT_OF_MEMORY, "" );
        return 1;
      }
    }
    SetReply( Result, ERR_NO_ERROR, "" );
    wsprintf( Buf, "Room size is %d bytes", VRoom.BufSz );
    AddToStringList( Result, Buf );
    return 1;
  }

FUNCDEF( ChatWR )
  {
    int  i;

    if( ! InitChat() ) {
      SetReply( Result, ERR_CHAT_INIT, "" );
      return 1;
    }
    LockRoom( &VRoom );
    for( i = 1; i < Params->Count; i++ )
      AddChatMessage( &VRoom, Params->Value[ i ] );
    UnlockRoom( &VRoom );
    SetReply( Result, ERR_NO_ERROR, "" );
    return 1;
  }

FUNCDEF( ChatWRnv )
  {
    int  i;

    if( ! InitChat() ) {
      SetReply( Result, ERR_CHAT_INIT, "" );
      return 1;
    }
    LockRoom( &NVRoom );
    for( i = 1; i < Params->Count; i++ )
      AddChatMessage( &NVRoom, Params->Value[ i ] );
    SaveRoom( &NVRoom );
    UnlockRoom( &NVRoom );
    SetReply( Result, ERR_NO_ERROR, "" );
    return 1;
  }
